<?php
/*
Plugin Name: AI Tarot Reader (78 Cards Fixed)
Description: فال تاروت کامل با 78 کارت و تعبیر مستقیم و معکوس از فایل JSON + جمع‌بندی کلی + محدودیت درخواست و ثبت موبایل
Version: 3.0
Author: mehrnaz tabari
*/

defined('ABSPATH') || exit;

// Load JS & CSS
add_action('wp_enqueue_scripts', function () {
    wp_enqueue_script('ai-tarot-js', plugin_dir_url(__FILE__) . 'ai-tarot.js', ['jquery'], null, true);
    wp_enqueue_style('ai-tarot-css', plugin_dir_url(__FILE__) . 'ai-tarot.css');

    wp_localize_script('ai-tarot-js', 'ai_tarot_ajax', [
        'ajax_url' => admin_url('admin-ajax.php'),
    ]);
});

// Shortcode to show the form
add_shortcode('ai_tarot_reader', function () {
    if (!is_user_logged_in()) {
        return '<div class="ai-tarot-error"><a href="https://erictarot.com/my-account/">لطفاً ابتدا وارد حساب کاربری خود شوید.</a></div>';
    }

    $user = wp_get_current_user();
    if (!in_array('customer', $user->roles) && !in_array('subscriber', $user->roles) && !current_user_can('administrator')) {
        return '<div class="ai-tarot-error">فقط کاربران دارای نقش "مشتری" یا "مشترک" به این بخش دسترسی دارند.</div>';
    }

    ob_start(); ?>
    <div class="ai-tarot">
        <input id="user_name" placeholder="نام شما">
        <input id="user_mobile" placeholder="شماره موبایل">
        <textarea id="user_question" placeholder="سؤال خود را بنویسید..."></textarea>
        <button id="draw_cards">گرفتن فال</button>
        <div id="tarot_result"></div>
    </div>
    <?php return ob_get_clean();
});

// Shortcode to view logs (admin only)
add_shortcode('ai_tarot_logs', function () {
    if (!current_user_can('administrator')) return;

    global $wpdb;
    $table = $wpdb->prefix . 'ai_tarot_logs';
    $logs = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC LIMIT 100");

    ob_start();
    echo "<table><tr><th>نام</th><th>موبایل</th><th>سؤال</th><th>زمان</th><th>حذف</th></tr>";
    foreach ($logs as $log) {
        echo "<tr>
            <td>{$log->name}</td>
            <td>{$log->mobile}</td>
            <td>{$log->question}</td>
            <td>{$log->created_at}</td>
            <td><a href='" . admin_url('admin-ajax.php?action=ai_tarot_delete_log&id=' . $log->id) . "' onclick='return confirm(\"حذف شود؟\")'>🗑️</a></td>
        </tr>";
    }
    echo "</table>";
    return ob_get_clean();
});

// AJAX
add_action('wp_ajax_ai_tarot_get_reading', 'ai_tarot_get_reading');
add_action('wp_ajax_ai_tarot_delete_log', function () {
    if (!current_user_can('administrator')) return;
    global $wpdb;
    $wpdb->delete($wpdb->prefix . 'ai_tarot_logs', ['id' => intval($_GET['id'])]);
    wp_redirect($_SERVER['HTTP_REFERER']);
    exit;
});

// Handle Tarot logic
function ai_tarot_get_reading()
{
    if (!is_user_logged_in()) {
        wp_send_json_error('باید وارد سایت شوید.');
    }

    $current_user = wp_get_current_user();
    $user_id = $current_user->ID;
    $name = sanitize_text_field($_POST['name'] ?? '');
    $mobile = sanitize_text_field($_POST['mobile'] ?? '');
    $question = sanitize_textarea_field($_POST['question'] ?? '');

    // نقش‌ها و موبایل ووکامرس
    if (!$mobile) {
        $mobile = get_user_meta($user_id, 'billing_phone', true) ?: get_user_meta($user_id, 'shipping_phone', true);
    }

    if (!$name || !$mobile) {
        wp_send_json_error('نام و شماره موبایل الزامی است.');
    }

    // محدودیت ۳ بار در ۲۴ ساعت (غیر از مدیر)
    if (!current_user_can('administrator')) {
        $limit_key = 'ai_tarot_usage_count';
        $time_key = 'ai_tarot_first_time';
        $count = (int) get_user_meta($user_id, $limit_key, true);
        $first_time = (int) get_user_meta($user_id, $time_key, true);
        $now = time();

        if (!$first_time || ($now - $first_time) > 86400) {
            update_user_meta($user_id, $limit_key, 1);
            update_user_meta($user_id, $time_key, $now);
        } elseif ($count >= 3) {
            wp_send_json_error('شما فقط ۳ بار در ۲۴ ساعت می‌توانید فال بگیرید.');
        } else {
            update_user_meta($user_id, $limit_key, $count + 1);
        }
    }

    // خواندن فایل کارت‌ها
    $json_path = plugin_dir_path(__FILE__) . 'tarot_meanings_78cards.json';
    if (!file_exists($json_path)) wp_send_json_error('فایل معانی کارت‌ها موجود نیست.');
    $meanings = json_decode(file_get_contents($json_path), true);
    if (!$meanings) wp_send_json_error('خطا در خواندن فایل معانی.');

    // انتخاب کارت‌ها
    $cards = array_keys($meanings);
    shuffle($cards);
    $selected = [];
    foreach (array_slice($cards, 0, 3) as $name_card) {
        $is_reversed = random_int(0, 1);
        $selected[] = ['name' => $name_card, 'reversed' => $is_reversed];
    }

    // جمع‌بندی بر اساس تعداد معکوس
    $rev_count = count(array_filter($selected, fn($c) => $c['reversed']));
    $summary_map = [
        0 => '🌈 نوید موفقیت، رشد و آرامش.',
        1 => '⚖️ ترکیبی از فرصت و مانع — نیاز به تصمیم‌گیری آگاهانه.',
        2 => '🔄 تقابل با درون یا تغییرات نزدیک.',
        3 => '👁️‍🗨️ دوره‌ای چالش‌برانگیز و نیاز به بازنگری.',
    ];

    $output = "<ul>";
    foreach ($selected as $c) {
        $n = $c['name'];
        $o = $c['reversed'] ? 'معکوس' : 'مستقیم';
        $m = $meanings[$n][$c['reversed'] ? 'reversed' : 'upright'];
        $output .= "<li><strong>$n ($o)</strong>:<br>$m</li><br>";
    }
    $output .= "</ul><div class='ai-tarot-summary'><strong>📌 جمع‌بندی:</strong> {$summary_map[$rev_count]}</div>";

    // ذخیره در DB
    global $wpdb;
    $table = $wpdb->prefix . 'ai_tarot_logs';
    $wpdb->insert($table, [
        'user_id' => $user_id,
        'name' => $name,
        'mobile' => $mobile,
        'question' => $question,
        'created_at' => current_time('mysql'),
    ]);

    wp_send_json_success([
        'reading' => $output,
        'cards' => array_map(fn($c) => $c['name'] . ($c['reversed'] ? ' (معکوس)' : ' (مستقیم)'), $selected)
    ]);
}

// Create DB table on plugin activation
register_activation_hook(__FILE__, function () {
    global $wpdb;
    $table = $wpdb->prefix . 'ai_tarot_logs';
    $charset = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE IF NOT EXISTS $table (
        id BIGINT AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT,
        name VARCHAR(255),
        mobile VARCHAR(20),
        question TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    ) $charset;";
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
});
